<?php

/*
 * @link              https://tebi.com
 * @package           tebi-reservations
 * @copyright         Copyright (c) Tebi
 * @license           GPL-2
 *
 * @wordpress-plugin
 * Plugin Name:       Tebi Reservations
 * Plugin URI:        https://tebi.com/en/reservations/
 * Description:       Add the Tebi Reservations widget to your Wordpress website.
 * Version:           1.1.1
 * Requires at least: 5.2
 * Requires PHP:      7.2
 * Author:            Tebi, Maciv
 * Author URI:        https://tebi.com/
 * Text Domain:       tebi-reservations
 * License:           GPL v2 or later
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 * Domain Path:       /languages
 */

if (!defined('ABSPATH'))
    exit;


class TebiReservationsWidget
{
    function __construct()
    {
        add_action('admin_menu', array($this, 'adminPage'));
        add_action('admin_init', array($this, 'settings'));
        add_action('wp_head', array($this, 'ifHead'));
        add_action('wp_footer', array($this, 'ifFooter'));
        add_action('init', array($this, 'languages'));
    }

    function languages()
    {
        load_plugin_textdomain('tebi-reservations', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    function ifHead()
    {
        if ($this->scriptLocation('0')) {
            $this->addScript();
        }
    }

    function ifFooter()
    {
        if ($this->scriptLocation('1')) {
            $this->addScript();
        }
    }

    function scriptLocation($location)
    {
        return (get_option('tebirw_token') != 'Enter Token') &&
            (get_option('tebirw_enable_widget') == '1') &&
            (get_option('tebirw_location') === $location);
    }

    function addScript()
    {
        echo '<script src="https://live.tebi.co/ecom/widget-manager.js" data-widget-token="' . esc_html(get_option('tebirw_token')) . '" id="tebi"></script>';
    }


    function settings()
    {
        add_settings_section('tebirw_main_section', null, 'displayReservations', 'tebi-reservations');

        add_settings_field('tebirw_token', esc_html__('Reservations Token', 'tebi-reservations'), array($this, 'tokenFieldHTML'), 'tebi-reservations', 'tebirw_main_section');
        register_setting('tebireservationswidget', 'tebirw_token', array('sanitize_callback' => array($this, 'sanitizeToken'), 'default' => 'Enter Token'));

        add_settings_field('tebirw_enable_widget', esc_html__('Activate Widget', 'tebi-reservations'), array($this, 'checkboxHTML'), 'tebi-reservations', 'tebirw_main_section', array('checkboxName' => 'tebirw_enable_widget'));
        register_setting('tebireservationswidget', 'tebirw_enable_widget', array('sanitize_callback' => 'sanitize_text_field', 'default' => '0'));

        add_settings_section('tebirw_advanced_section', null, null, 'tebi-reservations');

        add_settings_field('tebirw_location', esc_html__('Injection Location', 'tebi-reservations'), array($this, 'locationHTML'), 'tebi-reservations', 'tebirw_advanced_section');
        register_setting('tebireservationswidget', 'tebirw_location', array('sanitize_callback' => array($this, 'sanitizeLocation'), 'default' => '0'));

        function displayReservations()
        {
            esc_html_e('Instructions on how to set up the Tebi Reservations Widget.', 'tebi-reservations');
?>
            <p>
                1. <?php esc_html_e('Go to the', 'tebi-reservations'); ?>
                <a href="https://tebi.com/backoffice" target="_blank">Tebi Backoffice</a>
            </p>
            <p>
                2. <?php esc_html_e('Go to Reservations > Setup', 'tebi-reservations'); ?>.
            </p>
            <p>
                3. <?php esc_html_e('Copy your Tebi Reservations Token', 'tebi-reservations'); ?>.
            </p>
            <p>
                4. <?php esc_html_e('Paste your Reservations Token in the field below', 'tebi-reservations'); ?>.
            </p>
            <p>
                5. <?php esc_html_e('Click activate Widget', 'tebi-reservations'); ?>.
            </p>
            <p>
                6. <?php esc_html_e('Click Save Changes', 'tebi-reservations'); ?>.
            </p>
        <?php
        }
    }

    // function to sanatize tebirw_token which requires 6-8 numbers followed by _ and then the rest of the token

    function sanitizeToken($input)
    {
        if (preg_match('/^\d{6,8}_[a-zA-Z0-9]+$/', $input)) {
            return $input;
        } else {
            add_settings_error('tebirw_token', 'tebirw_token_error', esc_html__('The token does not seem to be correct. Check whether you have copied the correct token and try to enter it again. If that does not work please contact us.', 'tebi-reservations'));
            return get_option('tebirw_token');
        }
    }


    function sanitizeLocation($input)
    {
        if ($input != '0' and $input != '1') {
            add_settings_error('tebirw_location', 'tebirw_location_error', esc_html__('Injection Location must be either head or footer.', 'tebi-reservations'));
            return get_option('tebirw_location');
        }
        return $input;
    }

    function checkboxHTML($args)
    { ?>
        <input type="checkbox" name="<?php echo esc_html($args['checkboxName']); ?>" value="1" <?php checked(get_option($args['checkboxName']), '1') ?>>
    <?php }

    function tokenFieldHTML()
    { ?>
        <input type="text" name="tebirw_token" value="<?php echo esc_attr(get_option('tebirw_token')) ?>">
    <?php }

    function locationHTML()
    { ?>
        <select name="tebirw_location">
            <option value="0" <?php selected(get_option('tebirw_location'), '0') ?>>
                <?php esc_html_e('Head', 'tebi-reservations'); ?>
            </option>
            <option value="1" <?php selected(get_option('tebirw_location'), '1') ?>>
                <?php esc_html_e('Footer', 'tebi-reservations'); ?>
            </option>
        </select>
    <?php }



    function adminPage()
    {
        add_options_page(
            'Tebi Reservations Widget Settings',
            'Tebi Reservations',
            'manage_options',
            'tebi-reservations',
            array($this, 'adminPanelHTML')
        );
    }

    function adminPanelHTML()
    { ?>
        <div class="wrap">
            <h1>Tebi Reservations Widget</h1>
            <form action="options.php" method="POST">
                <?php
                settings_fields('tebireservationswidget');
                do_settings_sections('tebi-reservations');
                submit_button();
                ?>
            </form>
        </div>
<?php }
}

$TebiReservationsWidget = new TebiReservationsWidget();
